<?php
session_start();
include 'connect.php';

// Check if user is logged in and is an admin
if (!isset($_SESSION['user_id']) || !$_SESSION['is_admin']) {
    header("Location: login.php");
    exit();
}

// Handle car addition
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['add_car'])) {
    $car_name = $conn->real_escape_string($_POST['car_name']);
    $car_model = $conn->real_escape_string($_POST['car_model']);
    $car_price = $conn->real_escape_string($_POST['car_price']);
    $quantity = $conn->real_escape_string($_POST['quantity']);

    // Handle file upload
    $car_image = file_get_contents($_FILES['car_image']['tmp_name']);

    $stmt = $conn->prepare("INSERT INTO cars (car_name, car_model, car_price, car_image, quantity) VALUES (?, ?, ?, ?, ?)");
    $stmt->bind_param("siisi", $car_name, $car_model, $car_price, $car_image, $quantity);
    $stmt->execute();
    $stmt->close();
}

// Handle order approval/rejection
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['order_action'])) {
    $order_id = $conn->real_escape_string($_POST['order_id']);
    $action = $conn->real_escape_string($_POST['order_action']);

    $stmt = $conn->prepare("UPDATE orders SET status = ? WHERE id = ?");
    $stmt->bind_param("si", $action, $order_id);
    $stmt->execute();

    if ($action == 'Approved') {
        // Reduce car quantity by 1
        $stmt = $conn->prepare("UPDATE cars SET quantity = quantity - 1 WHERE id = (SELECT car_id FROM orders WHERE id = ?)");
        $stmt->bind_param("i", $order_id);
        $stmt->execute();
    }

    $stmt->close();
}

// Handle adding car specifications
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['add_specification'])) {
    $car_id = $conn->real_escape_string($_POST['car_id']);
    $specification = $conn->real_escape_string($_POST['specification']);
    $has_feature = $conn->real_escape_string($_POST['has_feature']);

    $stmt = $conn->prepare("INSERT INTO car_specifications (car_id, specification, has_feature) VALUES (?, ?, ?)");
    $stmt->bind_param("isi", $car_id, $specification, $has_feature);
    $stmt->execute();
    $stmt->close();
}

// Handle deleting car specifications
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['delete_specification'])) {
    $spec_id = $conn->real_escape_string($_POST['spec_id']);
    $stmt = $conn->prepare("DELETE FROM car_specifications WHERE id = ?");
    $stmt->bind_param("i", $spec_id);
    if ($stmt->execute()) {
        $success_message = "تم حذف المواصفات بنجاح";
    } else {
        $error_message = "حدث خطأ أثناء حذف المواصفات";
    }
    $stmt->close();
}

// Handle editing car specifications
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['edit_specification'])) {
    $spec_id = $conn->real_escape_string($_POST['spec_id']);
    $specification = $conn->real_escape_string($_POST['specification']);
    $has_feature = $conn->real_escape_string($_POST['has_feature']);

    $stmt = $conn->prepare("UPDATE car_specifications SET specification = ?, has_feature = ? WHERE id = ?");
    $stmt->bind_param("sii", $specification, $has_feature, $spec_id);
    if ($stmt->execute()) {
        $success_message = "تم تعديل المواصفات بنجاح";
    } else {
        $error_message = "حدث خطأ أثناء تعديل المواصفات";
    }
    $stmt->close();
}

// Handle editing car details
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['edit_car'])) {
    $car_id = $conn->real_escape_string($_POST['car_id']);
    $quantity = $conn->real_escape_string($_POST['quantity']);
    $price = $conn->real_escape_string($_POST['price']);
    
    $stmt = $conn->prepare("UPDATE cars SET quantity = ?, car_price = ? WHERE id = ?");
    $stmt->bind_param("iii", $quantity, $price, $car_id);
    $stmt->execute();
    $stmt->close();
}

// Fetch pending orders
$orders_query = "SELECT orders.*, cars.car_name, cars.car_model, users.username 
                 FROM orders 
                 JOIN cars ON orders.car_id = cars.id 
                 JOIN users ON orders.user_id = users.id 
                 WHERE orders.status = 'Pending'";
$orders_result = $conn->query($orders_query);

// Fetch all cars
$cars_query = "SELECT * FROM cars";
$cars_result = $conn->query($cars_query);

// Fetch all car specifications
$specs_query = "SELECT cs.*, c.car_name, c.car_model 
                FROM car_specifications cs
                JOIN cars c ON cs.car_id = c.id";
$specs_result = $conn->query($specs_query);
?>

<!DOCTYPE html>
<html lang="ar">

<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>لوحة التحكم الإدارية</title>
    <link rel="stylesheet" href="styles.css">
    <style>
        .admin-section {
            margin-bottom: 30px;
        }
        table {
            width: 100%;
            border-collapse: collapse;
        }
        th, td {
            border: 1px solid #ddd;
            padding: 8px;
            text-align: right;
        }
        th {
            background-color: #f2f2f2;
        }
        .car-specs {
            display: none;
        }
        .edit-form {
            display: none;
        }
    </style>
</head>

<body>
    <header>
        <h1>لوحة التحكم الإدارية</h1>
        <nav>
            <a href="index.php">الصفحة الرئيسية</a>
            <a href="admin_dashboard.php">لوحة التحكم</a>
            <a href="logout.php">تسجيل الخروج</a>
        </nav>
    </header>
    <main>
        <section class="admin-section">
            <h2>إضافة سيارة جديدة</h2>
            <form action="admin_dashboard.php" method="post" enctype="multipart/form-data">
                <label for="car_name">اسم السيارة:</label>
                <input type="text" id="car_name" name="car_name" required maxlength="50">
                
                <label for="car_model">موديل السيارة:</label>
                <input type="number" id="car_model" name="car_model" required max="9999">
                
                <label for="car_price">سعر السيارة:</label>
                <input type="number" id="car_price" name="car_price" required max="9999999999">
                
                <label for="quantity">الكمية:</label>
                <input type="number" id="quantity" name="quantity" required max="99999">
                
                <label for="car_image">صورة السيارة:</label>
                <input type="file" id="car_image" name="car_image" required accept="image/*">
                
                <button type="submit" name="add_car">إضافة السيارة</button>
            </form>
        </section>

        <section class="admin-section">
            <h2>الطلبات المعلقة</h2>
            <div style="overflow-x: auto;">
                <table>
                    <thead>
                        <tr>
                            <th>رقم الطلب</th>
                            <th>الاسم الكامل</th>
                            <th>اسم المستخدم</th>
                            <th>السيارة</th>
                            <th>تاريخ الحجز</th>
                            <th>تاريخ التسليم</th>
                            <th>رخصة القيادة</th>
                            <th>رقم الهوية الوطنية</th>
                            <th>السعر الإجمالي</th>
                            <th>الإجراء</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php while ($order = $orders_result->fetch_assoc()): ?>
                            <tr>
                                <td><?= $order['id'] ?></td>
                                <td><?= $order['fullname'] ?></td>
                                <td><?= $order['username'] ?></td>
                                <td><?= $order['car_name'] ?> <?= $order['car_model'] ?></td>
                                <td><?= $order['booking_date'] ?></td>
                                <td><?= $order['delivery_date'] ?></td>
                                <td><?= $order['driver_license'] ?></td>
                                <td><?= $order['national_id'] ?></td>
                                <td><?= $order['total_price'] ?> ريال</td>
                                <td>
                                    <form action="admin_dashboard.php" method="post">
                                        <input type="hidden" name="order_id" value="<?= $order['id'] ?>">
                                        <button type="submit" name="order_action" value="Approved">قبول</button>
                                        <button type="submit" name="order_action" value="Declined">رفض</button>
                                    </form>
                                </td>
                            </tr>
                        <?php endwhile; ?>
                    </tbody>
                </table>
            </div>
        </section>

        <section class="admin-section">
            <h2>قائمة السيارات والمواصفات</h2>
            <?php if (isset($success_message)): ?>
                <p class="success"><?php echo $success_message; ?></p>
            <?php endif; ?>
            <?php if (isset($error_message)): ?>
                <p class="error"><?php echo $error_message; ?></p>
            <?php endif; ?>
            <table>
                <thead>
                    <tr>
                        <th>السيارة</th>
                        <th>الموديل</th>
                        <th>السعر</th>
                        <th>الكمية</th>
                        <th>الإجراء</th>
                    </tr>
                </thead>
                <tbody>
                    <?php 
                    $cars_result->data_seek(0);
                    while ($car = $cars_result->fetch_assoc()): 
                    ?>
                        <tr>
                            <td><a href="#" onclick="toggleSpecs(<?= $car['id'] ?>); return false;"><?= $car['car_name'] ?></a></td>
                            <td><?= $car['car_model'] ?></td>
                            <td>$<?= $car['car_price'] ?></td>
                            <td><?= $car['quantity'] ?></td>
                            <td>
                                <form action="admin_dashboard.php" method="post">
                                    <input type="hidden" name="car_id" value="<?= $car['id'] ?>">
                                    <input type="number" name="quantity" value="<?= $car['quantity'] ?>" min="0" required>
                                    <input type="number" name="price" value="<?= $car['car_price'] ?>" min="0" required>
                                    <button type="submit" name="edit_car">تعديل</button>
                                </form>
                            </td>
                        </tr>
                        <tr id="specs-<?= $car['id'] ?>" class="car-specs">
                            <td colspan="5">
                                <h3>مواصفات السيارة</h3>
                                <table>
                                    <thead>
                                        <tr>
                                            <th>المواصفات</th>
                                            <th>متوفر</th>
                                            <th>الإجراء</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php
                                        $specs_result->data_seek(0);
                                        while ($spec = $specs_result->fetch_assoc()):
                                            if ($spec['car_id'] == $car['id']):
                                        ?>
                                            <tr>
                                                <td><?= $spec['specification'] ?></td>
                                                <td><?= $spec['has_feature'] ? 'نعم' : 'لا' ?></td>
                                                <td>
                                                    <button onclick="showEditForm(<?= $spec['id'] ?>)">تعديل</button>
                                                    <form action="admin_dashboard.php" method="post" onsubmit="return confirm('هل أنت متأكد من حذف هذه المواصفات؟');">
                                                        <input type="hidden" name="spec_id" value="<?= $spec['id'] ?>">
                                                        <button type="submit" name="delete_specification">حذف</button>
                                                    </form>
                                                </td>
                                            </tr>
                                            <tr id="edit-form-<?= $spec['id'] ?>" class="edit-form">
                                                <td colspan="3">
                                                    <form action="admin_dashboard.php" method="post">
                                                        <input type="hidden" name="spec_id" value="<?= $spec['id'] ?>">
                                                        <input type="text" name="specification" value="<?= $spec['specification'] ?>" required>
                                                        <select name="has_feature">
                                                            <option value="1" <?= $spec['has_feature'] ? 'selected' : '' ?>>نعم</option>
                                                            <option value="0" <?= !$spec['has_feature'] ? 'selected' : '' ?>>لا</option>
                                                        </select>
                                                        <button type="submit" name="edit_specification">حفظ التعديل</button>
                                                    </form>
                                                </td>
                                            </tr>
                                        <?php 
                                            endif;
                                        endwhile; 
                                        ?>
                                    </tbody>
                                    </table>
                                <h4>إضافة مواصفات جديدة</h4>
                                <form action="admin_dashboard.php" method="post">
                                    <input type="hidden" name="car_id" value="<?= $car['id'] ?>">
                                    <input type="text" name="specification" placeholder="المواصفات" required>
                                    <select name="has_feature">
                                        <option value="1">نعم</option>
                                        <option value="0">لا</option>
                                    </select>
                                    <button type="submit" name="add_specification">إضافة</button>
                                </form>
                            </td>
                        </tr>
                    <?php endwhile; ?>
                </tbody>
            </table>
        </section>
    </main>

    <script>
        function toggleSpecs(carId) {
            var specsRow = document.getElementById('specs-' + carId);
            if (specsRow.style.display === 'none' || specsRow.style.display === '') {
                specsRow.style.display = 'table-row';
            } else {
                specsRow.style.display = 'none';
            }
        }

        function showEditForm(specId) {
            var editForm = document.getElementById('edit-form-' + specId);
            if (editForm.style.display === 'none' || editForm.style.display === '') {
                editForm.style.display = 'table-row';
            } else {
                editForm.style.display = 'none';
            }
        }
    </script>
</body>

</html>